<?php
/*  
* Autor: rafael@clares.com.br - 2023
* $resizer = new ImageResizer('imagem_original.jpg', 800, 600, 'saveOrShow','newname');
*/

namespace PHPImager;

class ImageResizer
{
    private $imagePath;
    private $maxWidth;
    private $maxHeight;
    private $saveToDisk;

    public function __construct($imagePath, $maxWidth, $maxHeight, $saveToDisk = false)
    {
        $this->imagePath = $imagePath;
        $this->saveToDisk = $saveToDisk;
        $this->maxWidth = intval($maxWidth);
        $this->maxHeight = intval($maxHeight);
        $this->resize();
    }

    public function resize()
    {
        // Obtém as dimensões da imagem original
        $dimensions = getimagesize($this->imagePath);
        $originalWidth = $dimensions[0];
        $originalHeight = $dimensions[1];

        // Verifica se as dimensões originais são maiores que as informadas
        if ($originalWidth > $this->maxWidth || $originalHeight > $this->maxHeight) {
            // Calcula a proporção para manter a qualidade da imagem
            $ratio = min($this->maxWidth / $originalWidth, $this->maxHeight / $originalHeight);

            // Calcula as novas dimensões da imagem
            $newWidth = round($originalWidth * $ratio);
            $newHeight = round($originalHeight * $ratio);
        } else {
            $newWidth = $originalWidth;
            $newHeight = $originalHeight;
        }

        // Cria uma nova imagem com as dimensões redimensionadas
        $newImage = imagecreatetruecolor($newWidth, $newHeight);

        // Carrega a imagem original
        switch ($dimensions[2]) {
            case IMAGETYPE_JPEG:
                $originalImage = imagecreatefromjpeg($this->imagePath);
                break;
            case IMAGETYPE_PNG:
                $originalImage = imagecreatefrompng($this->imagePath);
                break;
            case IMAGETYPE_GIF:
                $originalImage = imagecreatefromgif($this->imagePath);
                break;
            case IMAGETYPE_WEBP:
                $originalImage = imagecreatefromwebp($this->imagePath);
                break;
            default:
                throw new \Exception('Tipo de imagem não suportado');
        }

        // Redimensiona a imagem original para a nova imagem
        imagecopyresampled($newImage, $originalImage, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);

        if ($this->saveToDisk) {
            // Salva a nova imagem com o mesmo formato da imagem original
            @copy($this->imagePath, $this->saveToDisk);
            $this->imagePath = $this->saveToDisk;

            switch ($dimensions[2]) {
                case IMAGETYPE_JPEG:
                    imagejpeg($newImage, $this->imagePath, 95);
                    break;
                case IMAGETYPE_PNG:
                    imagepng($newImage, $this->imagePath);
                    break;
                case IMAGETYPE_GIF:
                    imagegif($newImage, $this->imagePath);
                    break;
                case IMAGETYPE_WEBP:
                    imagewebp($newImage, $this->imagePath, 95);
                    break;
                default:
                    throw new \Exception('Tipo de imagem não suportado');
            }
        } else {
            // Exibe a nova imagem redimensionada na tela
            @header('Content-type: ' . image_type_to_mime_type($dimensions[2]));
            switch ($dimensions[2]) {
                case IMAGETYPE_JPEG:
                    imagejpeg($newImage);
                    break;
                case IMAGETYPE_PNG:
                    imagepng($newImage);
                    break;
                case IMAGETYPE_GIF:
                    imagegif($newImage);
                    break;
                case IMAGETYPE_WEBP:
                    imagewebp($newImage);
                    break;
                default:
                    throw new \Exception('Tipo de imagem não suportado');
            }
        }

        // Libera a memória das imagens
        imagedestroy($originalImage);
        imagedestroy($newImage);
    }

    static public function slug($string, $separator = '-')
    {
        $string = iconv('UTF-8', 'ASCII//TRANSLIT', $string);
        $string = preg_replace('/[^a-z0-9]/i', $separator, $string);
        $string = preg_replace('/' . preg_quote($separator) . '[' . preg_quote($separator) . ']+/', $separator, $string);
        $string = trim($string, $separator);
        return strtolower($string);
    }
}
