<?php
// Incluir arquivo com funções antiSQL
include_once('../antisql.php');

// Função para criar tabela automaticamente baseada nos campos do formulário
function createTableFromFormData($tableName, $formData, $uploadConfig = null) {
    global $conn;
    
    $fields = [];
    $fields[] = "`id` int(11) NOT NULL AUTO_INCREMENT PRIMARY KEY";
    
    foreach ($formData as $fieldName => $value) {
        $fieldType = 'VARCHAR(255)';
        
        // Definir tipos de campo baseado no nome
        if (in_array($fieldName, ['descricao', 'noticia', 'obs', 'anotacao', 'subdescricao'])) {
            $fieldType = 'TEXT';
        } elseif (in_array($fieldName, ['data', 'data_agenda', 'data_anotacao'])) {
            $fieldType = 'VARCHAR(20)';
        } elseif (in_array($fieldName, ['hora', 'hora_agenda', 'hora_anotacao'])) {
            $fieldType = 'VARCHAR(10)';
        } elseif (strpos($fieldName, 'id_') === 0) {
            $fieldType = 'INT(11)';
        } elseif (in_array($fieldName, ['porcentagem', 'preco'])) {
            $fieldType = 'DECIMAL(10,2)';
        } elseif (in_array($fieldName, ['foto', 'imagem', 'imagem2', 'imagem3', 'imagem4', 'imagem5', 'imagem6'])) {
            $fieldType = 'VARCHAR(500)';
        }
        
        $fields[] = "`$fieldName` $fieldType";
    }
    
    // Adicionar campos de upload se configurado
    if ($uploadConfig && !in_array($uploadConfig['field'], array_keys($formData))) {
        $fields[] = "`{$uploadConfig['field']}` VARCHAR(500)";
    }
    
    $createSQL = "CREATE TABLE `$tableName` (" . implode(', ', $fields) . ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
    
    $result = mysqli_query($conn, $createSQL);
    
    if ($result) {
        echo "<!-- Tabela '$tableName' criada automaticamente -->";
        return true;
    } else {
        echo "<!-- Erro ao criar tabela '$tableName': " . mysqli_error($conn) . " -->";
        return false;
    }
}

// Função para verificar se tabela existe
function tableExists($tableName) {
    global $conn;
    $result = mysqli_query($conn, "SHOW TABLES LIKE '$tableName'");
    return mysqli_num_rows($result) > 0;
}

// Função para obter as colunas de uma tabela
function getTableColumns($table) {
    global $conn;
    static $tableColumns = []; // Cache das colunas
    
    if (!isset($tableColumns[$table])) {
        $tableColumns[$table] = [];
        
        // Verificar se a tabela existe
        if (!tableExists($table)) {
            // Se a tabela não existe, retornar array vazio
            // A função createTableFromFormData será chamada depois
            return [];
        }
        
        $result = mysqli_query($conn, "DESCRIBE `$table`");
        if ($result) {
            while ($row = mysqli_fetch_assoc($result)) {
                $tableColumns[$table][] = $row['Field'];
            }
        }
    }
    
    return $tableColumns[$table];
}

// Função para construir query INSERT dinamicamente
function buildInsertQuery($table, $data, $excludeFields = []) {
    global $conn;
    
    $fields = [];
    $values = [];
    
    // Obter colunas reais da tabela
    $tableColumns = getTableColumns($table);
    
    // Campos que devem ser automaticamente excluídos do INSERT
    $autoExcludeFields = [
        'Inserir', 'Alterar', 'Excluir',  // Campos de ação
        'Submit', 'submit',               // Botões de envio
        'Reset', 'reset',                 // Botões de reset
        'Button', 'button',               // Outros botões
        'csrf_token', 'token',            // Tokens de segurança
        'MAX_FILE_SIZE'                   // Limite de upload
    ];
    
    $allExcludeFields = array_merge($autoExcludeFields, $excludeFields);
    
    foreach ($data as $key => $value) {
        // Verificar se o campo existe na tabela e não está na lista de exclusão
        if (in_array($key, $tableColumns) && !in_array($key, $allExcludeFields)) {
            $fields[] = "`$key`";
            // Para campos já processados por antiSQL*, não aplicar escape adicional
            if (in_array($key, ['descricao', 'noticia', 'obs', 'anotacao']) || 
                strpos($key, 'cor') !== false || strpos($key, '_cor') !== false) {
                // Valor já foi processado por função específica
                $values[] = "'" . $value . "'";
            } else {
                // Aplicar escape padrão
                $values[] = "'" . mysqli_real_escape_string($conn, $value) . "'";
            }
        }
    }
    
    if (empty($fields)) {
        return false; // Não há campos para inserir
    }
    
    return "INSERT INTO `$table` (" . implode(', ', $fields) . ") VALUES (" . implode(', ', $values) . ")";
}

// Função para construir query UPDATE dinamicamente
function buildUpdateQuery($table, $data, $id, $idField = 'id', $excludeFields = []) {
    global $conn;
    
    $sets = [];
    
    // Obter colunas reais da tabela
    $tableColumns = getTableColumns($table);
    
    // Lista de campos de ID comuns que devem ser excluídos automaticamente
    $commonIdFields = [
        'id_' . $table, // ex: id_equipe, id_foto, id_usuario
        'id' . $table,  // ex: idequipe, idfoto, idusuario
        $table . '_id'  // ex: equipe_id, foto_id
    ];
    
    // Campos que devem ser automaticamente excluídos do UPDATE
    $autoExcludeFields = [
        'Inserir', 'Alterar', 'Excluir',  // Campos de ação
        'Submit', 'submit',               // Botões de envio
        'Reset', 'reset',                 // Botões de reset
        'Button', 'button',               // Outros botões
        'csrf_token', 'token',            // Tokens de segurança
        'MAX_FILE_SIZE',                  // Limite de upload
        $idField                          // Campo ID principal
    ];
    
    $allExcludeFields = array_merge($autoExcludeFields, $excludeFields, $commonIdFields);
    
    foreach ($data as $key => $value) {
        // Verificar se o campo existe na tabela e não está na lista de exclusão
        if (in_array($key, $tableColumns) && !in_array($key, $allExcludeFields)) {
            // Para campos já processados por antiSQL*, não aplicar escape adicional
            if (in_array($key, ['descricao', 'noticia', 'obs', 'anotacao']) || 
                strpos($key, 'cor') !== false || strpos($key, '_cor') !== false) {
                // Valor já foi processado por função específica
                $sets[] = "`$key`='" . $value . "'";
            } else {
                // Aplicar escape padrão
                $sets[] = "`$key`='" . mysqli_real_escape_string($conn, $value) . "'";
            }
        }
    }
    
    if (empty($sets)) {
        return false; // Não há campos para atualizar
    }
    
    return "UPDATE `$table` SET " . implode(', ', $sets) . " WHERE `$idField`='" . mysqli_real_escape_string($conn, $id) . "'";
}

// Função para preparar dados do formulário dinamicamente
function prepareFormData($excludeFields = [], $autoFields = true, $tableName = null) {
    global $conn;
    
    $formData = [];
    
    // Campos que devem ser automaticamente excluídos
    $autoExcludeFields = [
        'Inserir', 'Alterar', 'Excluir',  // Campos de ação
        'Submit', 'submit',               // Botões de envio
        'Reset', 'reset',                 // Botões de reset
        'Button', 'button',               // Outros botões
        'csrf_token', 'token',            // Tokens de segurança
        'MAX_FILE_SIZE'                   // Limite de upload
    ];
    
    foreach ($_POST as $key => $value) {
        if (!in_array($key, array_merge($autoExcludeFields, $excludeFields))) {
            // Campos que precisam de escape especial
            if (in_array($key, ['descricao', 'noticia', 'obs', 'anotacao']) || 
                strpos($key, 'subtitulo') !== false || 
                strpos($key, 'pagina') !== false && strpos($key, '_subtitulo') !== false) {
                // Para campos de texto rico (TinyMCE), usar função específica
                $formData[$key] = antiSQLRichText($value);
            } elseif (strpos($key, 'cor') !== false || strpos($key, '_cor') !== false || strpos($key, 'color') !== false) {
                // Campos de cores - preservar # para cores hexadecimais
                $formData[$key] = antiSQLColor($value);
            } else {
                $formData[$key] = antiSQL($value);
            }
        }
    }
    
    // Adicionar campos automáticos apenas se a tabela tiver essas colunas
    if ($autoFields && $tableName && tableExists($tableName)) {
        $tableColumns = getTableColumns($tableName);
        if (in_array('data', $tableColumns)) {
            $formData['data'] = date("d/m/Y");
        }
        if (in_array('hora', $tableColumns)) {
            $formData['hora'] = date("H:i");
        }
    } elseif ($autoFields && !$tableName) {
        // Comportamento antigo - sempre adicionar se não especificar tabela
        $formData['data'] = date("d/m/Y");
        $formData['hora'] = date("H:i");
    } elseif ($autoFields && $tableName && !tableExists($tableName)) {
        // Se a tabela não existe ainda, adicionar campos básicos que provavelmente existirão
        $formData['data'] = date("d/m/Y");
        $formData['hora'] = date("H:i");
    }
    
    return $formData;
}

// Função para processar upload de arquivos automaticamente
function processFileUpload($fileFieldName = 'foto', $uploadDir = 'uploads', $allowedExtensions = ['.jpg', '.jpeg', '.gif', '.png', '.pdf', '.doc', '.docx', '.txt']) {
    if (!isset($_FILES[$fileFieldName]['name']) || $_FILES[$fileFieldName]["error"] != 0) {
        return false;
    }

    $arquivo_tmp = $_FILES[$fileFieldName]['tmp_name'];
    $nomeArquivo = $_FILES[$fileFieldName]['name'];

    // Pega a extensão
    $extensao = strrchr($nomeArquivo, '.');
    $extensao = strtolower($extensao);

    // Verificar extensões permitidas
    $extensoesPermitidas = implode(';', $allowedExtensions) . ';';
    if (!strstr($extensoesPermitidas, $extensao)) {
        echo "Arquivos permitidos são: \"" . implode(';', $allowedExtensions) . "\"<br />";
        return false;
    }

    // Cria um nome único para o arquivo
    $novoNome = md5(microtime()) . $extensao;
    $caminhoCompleto = "../{$uploadDir}/" . $novoNome;

    // Criar diretório se não existir
    if (!file_exists("../{$uploadDir}/")) {
        mkdir("../{$uploadDir}/", 0755, true);
    }

    // Tenta mover o arquivo
    if (@move_uploaded_file($arquivo_tmp, $caminhoCompleto)) {
        // Remove o '../' para não dar conflito no ADMIN
        return str_replace('../', '', $caminhoCompleto);
    } else {
        echo "Erro ao salvar o arquivo. Aparentemente você não tem permissão de escrita.<br />";
        return false;
    }
}

// Função para processar múltiplos uploads
function processMultipleUploads($fileFields = [], $uploadDir = 'uploads', $allowedExtensions = ['.jpg', '.jpeg', '.gif', '.png', '.pdf', '.doc', '.docx', '.txt']) {
    $uploadedFiles = [];
    
    foreach ($fileFields as $fieldName) {
        if (isset($_FILES[$fieldName]['name']) && $_FILES[$fieldName]["error"] == 0) {
            $arquivo_tmp = $_FILES[$fieldName]['tmp_name'];
            $nomeArquivo = $_FILES[$fieldName]['name'];

            // Pega a extensão
            $extensao = strrchr($nomeArquivo, '.');
            $extensao = strtolower($extensao);

            // Verificar extensões permitidas
            $extensoesPermitidas = implode(';', $allowedExtensions) . ';';
            if (strstr($extensoesPermitidas, $extensao)) {
                // Cria um nome único
                $novoNome = md5(microtime() . $fieldName) . $extensao;
                $caminhoCompleto = "../{$uploadDir}/" . $novoNome;

                // Criar diretório se não existir
                if (!file_exists("../{$uploadDir}/")) {
                    mkdir("../{$uploadDir}/", 0755, true);
                }

                // Tenta mover o arquivo
                if (@move_uploaded_file($arquivo_tmp, $caminhoCompleto)) {
                    // Remove o '../' para não dar conflito no ADMIN
                    $uploadedFiles[$fieldName] = str_replace('../', '', $caminhoCompleto);
                } else {
                    echo "Erro ao salvar o arquivo {$fieldName}.<br />";
                }
            } else {
                echo "Extensões permitidas para {$fieldName}: " . implode(', ', $allowedExtensions) . "<br />";
            }
        }
    }
    
    return $uploadedFiles;
}

// Função para processar operações em tabelas de configuração (um registro fixo)
function processConfigOperations($tableName, $uploadConfig = null, $excludeFields = [], $redirectFile = null) {
    global $conn;
    
    $Alterar = antiSQL($_POST['Alterar'] ?? '');
    
    // Preparar dados dinâmicos do formulário (excluindo ID)
    $formData = prepareFormData(['id'], true, $tableName);
    
    // Determinar nome do arquivo para redirecionamento
    $redirectName = $redirectFile ?? $tableName;
    
    // Verificar se a tabela existe
    if (!tableExists($tableName)) {
        echo "<!-- Tabela '$tableName' não existe. Criando automaticamente... -->";
        createTableFromFormData($tableName, $formData, $uploadConfig);
        
        // Para tabelas de configuração, inserir o primeiro registro com ID = 0
        $initialData = [];
        foreach ($formData as $field => $value) {
            $initialData[$field] = '';
        }
        $insertQuery = buildInsertQuery($tableName, $initialData, $excludeFields);
        if ($insertQuery !== false) {
            mysqli_query($conn, $insertQuery);
            // Definir ID como 0 para tabelas de configuração
            mysqli_query($conn, "UPDATE `{$tableName}` SET `id` = 0 WHERE `id` = LAST_INSERT_ID()");
        }
    }
    
    // Verificar se existe o registro de configuração (ID = 0)
    $checkConfig = mysqli_query($conn, "SELECT COUNT(*) as total FROM `{$tableName}` WHERE `id` = 0");
    $configExists = mysqli_fetch_assoc($checkConfig)['total'] > 0;
    
    // Se não existe, criar o registro de configuração
    if (!$configExists) {
        $initialData = [];
        foreach ($formData as $field => $value) {
            $initialData[$field] = '';
        }
        $initialData['id'] = 0;
        $insertQuery = buildInsertQuery($tableName, $initialData, $excludeFields);
        if ($insertQuery !== false) {
            mysqli_query($conn, str_replace('INSERT INTO', 'INSERT IGNORE INTO', $insertQuery));
        }
    }
    
    // Processar upload se configurado
    $uploadedFile = false;
    if ($uploadConfig && ($_POST && $Alterar == "ok")) {
        $uploadedFile = processFileUpload(
            $uploadConfig['field'] ?? 'foto',
            $uploadConfig['dir'] ?? "uploads/{$tableName}",
            $uploadConfig['extensions'] ?? ['.jpg', '.jpeg', '.gif', '.png', '.pdf', '.doc', '.docx', '.txt']
        );
    }
    
    // UPDATE - Sempre atualiza o registro ID = 0
    if ($Alterar == "ok") {
        if ($uploadedFile) {
            $formData[$uploadConfig['field'] ?? 'foto'] = $uploadedFile;
        }
        
        $query = buildUpdateQuery($tableName, $formData, 0, 'id', $excludeFields);
        
        if ($query === false) {
            echo "Erro: Nenhum campo válido para atualizar.";
            exit();
        }
        
        $resultado = mysqli_query($conn, $query);
        
        if ($resultado) {
            header("Location: {$redirectName}.php?alterado_sucesso=sim");
            exit();
        } else {
            echo "Erro na query UPDATE: " . mysqli_error($conn);
            exit();
        }
    }
    
    return [
        'acao' => 'alterar',
        'recordId' => 0,
        'formData' => $formData
    ];
}

// Função para processar operações CRUD automaticamente
function processCRUDOperations($tableName, $idFieldName = 'id', $uploadConfig = null, $excludeFields = []) {
    global $conn;
    
    $acao = antiSQL($_GET['acao'] ?? '');
    $recordId = antiSQL($_GET[$idFieldName] ?? $_POST[$idFieldName] ?? '');
    $Inserir = antiSQL($_POST['Inserir'] ?? '');
    $Alterar = antiSQL($_POST['Alterar'] ?? '');
    $Excluir = antiSQL($_GET['Excluir'] ?? '');

    // Preparar dados dinâmicos do formulário
    $formData = prepareFormData([$idFieldName, 'id' . $tableName], true, $tableName);

    // Verificar se a tabela existe, se não existir, criar automaticamente
    if (!tableExists($tableName)) {
        echo "<!-- Tabela '$tableName' não existe. Criando automaticamente... -->";
        createTableFromFormData($tableName, $formData, $uploadConfig);
        
        // Limpar cache de colunas para forçar reload
        static $tableColumns = [];
        unset($tableColumns[$tableName]);
    }

    // Processar upload se configurado
    $uploadedFiles = [];
    if ($uploadConfig && ($_POST && ($Inserir == "ok" || $Alterar == "ok"))) {
        if (isset($uploadConfig['fields']) && is_array($uploadConfig['fields'])) {
            // Múltiplos uploads
            $uploadedFiles = processMultipleUploads(
                $uploadConfig['fields'],
                $uploadConfig['dir'] ?? "uploads/{$tableName}",
                $uploadConfig['extensions'] ?? ['.jpg', '.jpeg', '.gif', '.png', '.pdf', '.doc', '.docx', '.txt']
            );
        } else {
            // Upload único (compatibilidade)
            $uploadedFile = processFileUpload(
                $uploadConfig['field'] ?? 'foto',
                $uploadConfig['dir'] ?? "uploads/{$tableName}",
                $uploadConfig['extensions'] ?? ['.jpg', '.jpeg', '.gif', '.png', '.pdf', '.doc', '.docx', '.txt']
            );
            if ($uploadedFile) {
                $uploadedFiles[$uploadConfig['field'] ?? 'foto'] = $uploadedFile;
            }
        }
    }

    // INSERT
    if ($Inserir == "ok") {
        // Adicionar arquivos enviados ao formData
        foreach ($uploadedFiles as $field => $file) {
            $formData[$field] = $file;
        }
        
        $query = buildInsertQuery($tableName, $formData, $excludeFields);
        
        if ($query === false) {
            echo "Erro: Nenhum campo válido para inserir.";
            exit();
        }
        
        $resultado = mysqli_query($conn, $query);
        
        if ($resultado) {
            header("Location: {$tableName}.php?cadastrado_sucesso=sim");
            exit();
        } else {
            echo "Erro na query INSERT: " . mysqli_error($conn);
            exit();
        }
    }

    // Atualizar arquivos separadamente se foram enviados
    if (!empty($uploadedFiles) && !empty($recordId)) {
        foreach ($uploadedFiles as $fieldName => $uploadedFile) {
            // Excluir arquivo antigo
            $excluiarq_antigo = mysqli_query($conn, "SELECT {$fieldName} FROM {$tableName} WHERE id='$recordId'")->fetch_assoc();
            if ($excluiarq_antigo[$fieldName]) {
                @unlink($excluiarq_antigo[$fieldName]);
            }
            
            $updateFile = buildUpdateQuery($tableName, [$fieldName => $uploadedFile], $recordId, 'id');
            if ($updateFile !== false) {
                mysqli_query($conn, $updateFile);
            }
        }
    }

    // UPDATE
    if ($Alterar == "ok" && !empty($recordId)) {
        // Excluir campos de arquivo da atualização principal (já foram tratados acima)
        $fileFields = [];
        if ($uploadConfig) {
            if (isset($uploadConfig['fields'])) {
                $fileFields = $uploadConfig['fields'];
            } else {
                $fileFields = [$uploadConfig['field'] ?? 'foto'];
            }
        }
        $allExcludeFields = array_merge($excludeFields, $fileFields);
        
        $query = buildUpdateQuery($tableName, $formData, $recordId, 'id', $allExcludeFields);
        
        if ($query === false) {
            echo "Erro: Nenhum campo válido para atualizar.";
            exit();
        }
        
        $resultado = mysqli_query($conn, $query);
        
        if ($resultado) {
            header("Location: {$tableName}.php?alterado_sucesso=sim");
            exit();
        } else {
            echo "Erro na query UPDATE: " . mysqli_error($conn);
            exit();
        }
    }

    // DELETE
    if ($Excluir == "ok" && !empty($recordId)) {
        // Excluir arquivos antes de excluir o registro
        if ($uploadConfig) {
            $fileFields = [];
            if (isset($uploadConfig['fields'])) {
                $fileFields = $uploadConfig['fields'];
            } else {
                $fileFields = [$uploadConfig['field'] ?? 'foto'];
            }
            
            foreach ($fileFields as $fieldName) {
                $excluiarq_antigo = mysqli_query($conn, "SELECT {$fieldName} FROM {$tableName} WHERE id='$recordId'")->fetch_assoc();
                if ($excluiarq_antigo[$fieldName]) {
                    @unlink($excluiarq_antigo[$fieldName]);
                }
            }
        }
        
        $deleteQuery = "DELETE FROM `{$tableName}` WHERE `id`='" . mysqli_real_escape_string($conn, $recordId) . "'";
        $resultado = mysqli_query($conn, $deleteQuery);
        
        if ($resultado) {
            header("Location: {$tableName}.php?deletado_sucesso=sim");
            exit();
        } else {
            echo "Erro na query DELETE: " . mysqli_error($conn);
            exit();
        }
    }

    return [
        'acao' => $acao,
        'recordId' => $recordId,
        'formData' => $formData
    ];
}
?>